-----------------------------------------------------------------------------
--
--  This library is free software; you can redistribute it and/or
--  modify it under the terms of the GNU General Public
--  License as published by the Free Software Foundation; either
--  version 2 of the License, or (at your option) any later version.
--
-----------------------------------------------------------------------------   
-- Entity:      pcxahb
-- File:        pcxahb.vhd
-- Author:      Jiri Gaisler, Martin Johansson - Gaisler Research
-- Description: PCX/AHB bridge for T1 processor
----------------------------------------------------------------------------- 

library ieee;
use ieee.std_logic_1164.all;
library grlib;
use grlib.amba.all;
use grlib.stdlib.all;
use grlib.devices.all;
library gaisler;
use gaisler.misc.all;
use work.t1.all;
library techmap;
use techmap.gencomp.all;

entity pcxahb is
   generic (
     hindex : integer := 0;
     pindex : integer := 0;
     paddr  : integer := 0;
     pmask  : integer := 16#fff#;
     pirq   : integer := 0;
     pow    : integer := 1	-- Start on power-on
	);
   port (
      rst   : in  std_logic;
      clk   : in  std_ulogic;
      t1in  : out t1_in_type;
      t1out : in  t1_out_type;
      apbi  : in  apb_slv_in_type;
      apbo  : out apb_slv_out_type;
      ahbi  : in  ahb_mst_in_type;
      ahbo  : out ahb_mst_out_type 
      );
end;      

architecture struct of pcxahb is

constant pconfig : apb_config_type := (
  0 => ahb_device_reg ( VENDOR_GAISLER, GAISLER_T1AHB, 0, 0, pirq),
  1 => apb_iobar(paddr, pmask));

constant dbuf   : integer := 8;

-- PCX request types
constant LOAD_RQ    : std_logic_vector(4 downto 0) := "00000"; -- Load
constant STORE_RQ   : std_logic_vector(4 downto 0) := "00001"; -- Store
constant CAS1_RQ    : std_logic_vector(4 downto 0) := "00010"; -- Cas(1)
constant CAS2_RQ    : std_logic_vector(4 downto 0) := "00011"; -- Cas(2)
constant STRLOAD_RQ : std_logic_vector(4 downto 0) := "00100"; -- Stream loads
constant STRST_RQ   : std_logic_vector(4 downto 0) := "00101"; -- Stream Store
constant SWAP_RQ    : std_logic_vector(4 downto 0) := "00110"; -- SWP Ldstb
constant STQ_RQ	    : std_logic_vector(4 downto 0) := "00111"; -- special: only used for verification
constant INT_RQ	    : std_logic_vector(4 downto 0) := "01001"; -- Int
constant FP_1	    : std_logic_vector(4 downto 0) := "01010"; -- floating point: FP(1)
constant FP_2	    : std_logic_vector(4 downto 0) := "01011"; -- floating point: FP(2)
constant FWD_RQ	    : std_logic_vector(4 downto 0) := "01101"; -- Fwd req
constant FWD_RPY    : std_logic_vector(4 downto 0) := "01110"; -- Fwd reply
constant IMISS_RQ   : std_logic_vector(4 downto 0) := "10000"; -- Ifill Req
constant RSVD_RQ    : std_logic_vector(4 downto 0) := "11111"; -- special: reserved

-- CPX response types
constant LOAD_RET   : std_logic_vector(3 downto 0) := "0000"; -- Load
constant IFILL_RET  : std_logic_vector(3 downto 0) := "0001"; -- I$fill (1) L2,IOB","I$fill (2) L2
constant STRLOAD_RET: std_logic_vector(3 downto 0) := "0010"; -- Strm Load
constant INV_RET    : std_logic_vector(3 downto 0) := "0011"; -- Evict Inv
constant ST_ACK	    : std_logic_vector(3 downto 0) := "0100"; -- Store ACK
constant TEST_RET   : std_logic_vector(3 downto 0) := "0101"; -- special: only used for verification
constant STRST_ACK  : std_logic_vector(3 downto 0) := "0110"; -- Strm Store ACK
constant INT_RET    : std_logic_vector(3 downto 0) := "0111"; -- Int
constant FP_RET	    : std_logic_vector(3 downto 0) := "1000"; -- FP
constant FWD_RQ_RET : std_logic_vector(3 downto 0) := "1010"; -- Fwd req
constant FWD_RPY_RET: std_logic_vector(3 downto 0) := "1011"; -- Fwd Reply
constant ERR_RET    : std_logic_vector(3 downto 0) := "1100"; -- Error
constant RSVD_RET   : std_logic_vector(3 downto 0) := "1111"; -- special: reserved

-- CPX shared bits
constant CPX_IF4B   : integer := 1; 
constant CPX_IFILL2 : integer := 0;
constant CPX_ATOMIC : integer := 0;

type dma_state_type is (readc, writec);
subtype word32 is std_logic_vector(31 downto 0);
type datavec is array (0 to dbuf-1) of word32;
-- 0-123 = pcx packet 1, 124 = atomic transfer?, 125-188 data of packet 2
subtype pcxvec is std_logic_vector(188 downto 0);
constant fdepth : integer := 4;  -- Depth of pcx fifo, must be 2**n aligned
type pcxarr is array (0 to fdepth-1) of pcxvec;
constant fbits : integer := log2x(fdepth);

-- Bus states for Amba
constant A_INIT  : std_logic_vector(2 downto 0) := "000";
constant A_ONE   : std_logic_vector(2 downto 0) := "001";
constant A_IDLE  : std_logic_vector(2 downto 0) := "010";
constant A_REQ1  : std_logic_vector(2 downto 0) := "011";
constant A_REQ2  : std_logic_vector(2 downto 0) := "100";
constant A_REQ3  : std_logic_vector(2 downto 0) := "101";
constant A_ERR   : std_logic_vector(2 downto 0) := "110";

-- Bus states for PCX
constant P_IDLE	  : std_logic_vector(2 downto 0) := "000";
constant P_PQ	  : std_logic_vector(2 downto 0) := "001";
constant P_PA	  : std_logic_vector(2 downto 0) := "010";
constant P_PX	  : std_logic_vector(2 downto 0) := "011";
constant P_PX2	  : std_logic_vector(2 downto 0) := "100";
-- On powerup, before T1 is initialized
constant P_INIT   : std_logic_vector(2 downto 0) := "101";

-- Type of packet that should be sent on cpx
constant C_NONE	    : std_logic_vector(1 downto 0) := "00";
constant C_NORMAL   : std_logic_vector(1 downto 0) := "01";
constant C_INVDCACHE: std_logic_vector(1 downto 0) := "10";
constant C_ATOMIC2  : std_logic_vector(1 downto 0) := "11";

type reg_type is record
  abstate : std_logic_vector(2 downto 0); -- Amba bus_state_type;

  cpxtype : std_logic_vector(3 downto 0);
  cpxdata : std_logic_vector(127 downto 0);
  cpxsh   : std_logic_vector(4 downto 0); -- Shared bits
  cpxnc   : std_logic;			  -- Not cacheable
  cpxpfl  : std_logic;			  -- Reserved bit, used for Prefetch Load

  cpxrdy  : std_logic;

  pbstate : std_logic_vector(2 downto 0); -- PCX bus state
  pcxrq   : std_logic_vector(4 downto 0);
  oldpcxrq: std_logic_vector(4 downto 0);
  pcxreq  : std_logic_vector(4 downto 0);
  pcxgnt  : std_logic_vector(4 downto 0);
  pcxdata2: std_logic_vector(63 downto 0);
  pcxdata : std_logic_vector(63 downto 0);
  pcxaddr : std_logic_vector(39 downto 0);
  pcxsize : std_logic_vector(2 downto 0);
  pcxcpu  : std_logic_vector(2 downto 0);
  pcxthre : std_logic_vector(1 downto 0);
  pcxway  : std_logic_vector(1 downto 0);
  pcxbis  : std_logic;
  pcxinv  : std_logic;
  pcxpf	  : std_logic;
  pcxatom : std_logic;
  pcxnc   : std_logic;  -- Not cacheable
  pcxnew  : std_logic;
  pcxat   : std_logic;
  oldpcxat: std_logic;

  ioaddr  : std_logic; -- Address is I/O ?
  srcaddr : std_logic_vector(31 downto 0);
  srcinc  : std_logic_vector(1 downto 0);
  len     : std_logic_vector(15 downto 0);
  enable  : std_logic;
  write   : std_logic;
  inhibit : std_logic;
  ld2op   : std_logic;
  ld2val  : std_logic;
  op2     : std_logic_vector(123 downto 0);
  req2    : std_logic_vector(4 downto 0);
  status  : std_logic_vector(1 downto 0);
  data    : datavec;
  cnt, size : integer range 0 to dbuf-1;
  waddr, raddr  : std_logic_vector(1 downto 0);
  pcxbuf : pcxarr;
  reset_l : std_logic;
  doint   : std_logic;
  intcmd  : std_logic_vector(17 downto 0);

  atom2    : std_logic; -- High when iterating A_* the second time for atomics
  atomstore: std_logic; -- Whether CAS/SWAP should perform store or not
  storedata: std_logic_vector(63 downto 0);

  oldpcxthre     : std_logic_vector(1 downto 0);
  packettypecpx : std_logic_vector(1 downto 0);
end record;

signal r, rin : reg_type;
signal dmai : ahb_dma_in_type;
signal dmao : ahb_dma_out_type;

begin

  comb : process(apbi, dmao, rst, r, t1out)
  variable v       : reg_type;
  variable regd    : std_logic_vector(31 downto 0);   -- Data from registers
  variable start   : std_logic;
  variable burst   : std_logic;
  variable write   : std_logic;
  variable ready   : std_logic;
  variable retry   : std_logic;
  variable mexc    : std_logic;
  variable irq     : std_logic;
  variable size    : std_logic_vector( 1 downto 0);   -- DMA transfer size
  variable newlen  : std_logic_vector(15 downto 0);
  variable oldaddr : std_logic_vector(9 downto 0);
  variable newaddr : std_logic_vector(9 downto 0);
  variable oldsize : std_logic_vector( 1 downto 0);
  variable ainc    : std_logic_vector( 3 downto 0);
  variable empty   : std_logic;
  variable fspc_pcx_data_pa  : pcxvec;
  variable fread   : std_logic;
  variable rdata   : std_logic_vector(15 downto 0);
  variable reset_l : std_logic;  -- Local reset 

  begin

    v := r; regd := (others => '0'); burst := '0'; start := '0';
    write := '0'; ready := '0'; mexc := '0';
    irq := '0'; v.inhibit := '0';
    newlen := r.len - 1;
    rdata :=  r.data(0)(31 downto 16);

    -- PCX fifo with pass through (for non atomic packets)
    if (r.oldpcxrq /= "00000") and (r.oldpcxat = '1') then
      -- Second packet in atomic transfer
      -- Only data-field is needed from the second packet
      v.pcxbuf(conv_integer(r.waddr(fbits-1 downto 0)))(188 downto 125) := t1out.spc_pcx_data_pa(63 downto 0);
      v.pcxgnt := r.oldpcxrq;
      v.waddr := r.waddr + 1;
      v.pcxnew := '0';
    elsif (r.pcxrq /= "00000") then
      -- Single packet or first packet in an atomic transfer
      v.pcxbuf(conv_integer(r.waddr(fbits-1 downto 0)))(124 downto 0) := r.pcxat & t1out.spc_pcx_data_pa;
      v.pcxgnt := r.pcxrq;
      if (r.pcxat = '0') then
	-- Single packet transfer
	v.waddr := r.waddr + 1;
	v.pcxnew := '1'; -- pass through for non atomics
      else
	-- Atomic transfer, no pass through
	v.pcxnew := '0';
      end if;
    else
      v.pcxgnt := "00000";
      v.pcxnew := '0';
    end if;
    -- Pass through or use data from fifo
    fspc_pcx_data_pa := (others => '0');
    if (r.waddr = r.raddr) then
      -- Pass through (if not atomic)
      empty := '1';
      -- Bits 188-125 are not valid but these are not used for non atomic transfers
      fspc_pcx_data_pa(124 downto 0) := r.pcxat & t1out.spc_pcx_data_pa;
    else
      -- Use data from buffer
      empty := '0';
      fspc_pcx_data_pa := r.pcxbuf(conv_integer(r.raddr(fbits-1 downto 0)));
    end if;
    -- Save older request and atomic info
    v.pcxrq    := t1out.spc_pcx_req_pq;
    v.oldpcxrq := r.pcxrq;
    v.pcxat    := t1out.spc_pcx_atom_pq;
    v.oldpcxat := r.pcxat;

    fread := '0'; v.cpxrdy := '0';
    v.packettypecpx := C_NONE;

    case r.abstate is
    when A_INIT =>	-- Power-on reset
      if pow = 1 then 
        v.reset_l := '1'; -- Start on power-on
      end if;
      v.waddr := (others => '0'); v.raddr := (others => '0');
      if (t1out.rst_ready = '1') then 
	v.abstate := A_ONE;
        -- VINT is duplicated
        v.cpxdata(127 downto 82) := (others => '0');
	v.cpxdata(81 downto 64)  := "01" & X"0001"; -- VINT
        v.cpxdata(63 downto 18)  := (others => '0');
	v.cpxdata(17 downto 0)   := "01" & X"0001"; -- VINT
        v.cpxtype := INT_RET;	-- INT
        v.cpxrdy := '1';
      end if;
    when A_ONE  =>	-- Send one cpx packet to processor
      v.abstate := A_IDLE;
      v.packettypecpx := C_NORMAL;
    when A_IDLE =>	-- Look for new pcx packet and set up AHB
      if r.doint = '1' then
	v.abstate := A_ONE;
        -- VINT is duplicated
        v.cpxdata(127 downto 82) := (others => '0');
	v.cpxdata(81 downto 64)  := r.intcmd; -- VINT
        v.cpxdata(63 downto 18)  := (others => '0');
	v.cpxdata(17 downto 0)   := r.intcmd; -- VINT
        v.cpxtype := INT_RET;
        v.cpxrdy := '1';
      elsif (empty = '0') or (r.pcxnew = '1') then
	-- PCX packet fields
	v.pcxdata2:= fspc_pcx_data_pa(188 downto 125); -- Only needed for atomic transfers
        v.pcxatom := fspc_pcx_data_pa(124); --spc_pcx_atom_pq;
        v.pcxreq  := fspc_pcx_data_pa(122 downto 118);
        v.pcxnc   := fspc_pcx_data_pa(117);
        v.pcxcpu  := fspc_pcx_data_pa(116 downto 114);
        v.pcxthre := fspc_pcx_data_pa(113 downto 112);
        v.pcxinv  := fspc_pcx_data_pa(111);
	v.pcxpf	  := fspc_pcx_data_pa(110); -- prefetch ("Block Store" for Store)
        v.pcxbis  := fspc_pcx_data_pa(109);
	v.pcxway  := fspc_pcx_data_pa(108 downto 107); -- Replace L1 Way
        v.pcxsize := fspc_pcx_data_pa(106 downto 104);
        v.pcxaddr := fspc_pcx_data_pa(103 downto 64);
        v.pcxdata := fspc_pcx_data_pa(63 downto 0);
	fread := '1';
	if (v.pcxreq = CAS1_RQ) and (v.pcxsize = "011") then
	  -- Extended word
	  v.storedata := v.pcxdata2;
	elsif v.pcxreq = CAS1_RQ then
	  -- Word (the size of CAS packet is either Extended Word or Word)
	  -- Word data for store in CAS operations comes in a strange form
	  -- The lower 32 bits are not OK
	  v.storedata(63 downto 32) := v.pcxdata2(63 downto 32);
	  v.storedata(31 downto 0)  := v.pcxdata2(63 downto 32);
	else
	  -- Other requests that write to memory uses data from first pcx packet
	  v.storedata := v.pcxdata;
	end if;
	v.cnt := 0;
        v.cpxsh := (others => '0');
	v.cpxpfl := '0';
	v.atomstore := '1';
	-- General variables generated from PCX packet fields
	v.srcaddr := v.pcxaddr(31 downto 0);
	v.ioaddr := v.pcxaddr(39); -- MSB of pcxaddr indicates if pcx is memory mapped I/O
        if v.pcxsize(1 downto 0) = "11" then
	  v.srcinc := "10";
          if (v.pcxsize(2) = '1') then v.size := 3;
	  else v.size := 1; end if;
        else 
	  v.srcinc := v.pcxsize(1 downto 0); v.size := 0;
        end if;
	v.cpxnc := v.ioaddr; -- I/O is not cacheable
	-- Setup for each request
        case v.pcxreq is
        when LOAD_RQ | CAS1_RQ | SWAP_RQ =>	
	  -- The CAS/SWAP code is only for the LOAD part as Amba never goes IDLE in between LOAD/STORE parts
	  if v.pcxpf = '1' then
	    -- Prefetches are not cachable, "Prefetch Load" should also be asserted
	    v.cpxpfl := '1';
	    v.cpxnc := '1';
	  end if;
	  if (v.pcxreq = CAS1_RQ) or (v.pcxreq = SWAP_RQ) then
	    v.cpxsh(CPX_ATOMIC) := '1';
	    v.cpxnc := '1';
	  end if;
	  if v.ioaddr = '0' then 
	    v.size := 3; v.srcinc := "10"; v.srcaddr(3 downto 0) := "0000";
	  end if; 
	  v.enable := '1'; v.write := '0'; v.abstate := A_REQ2;
        when IMISS_RQ =>	
          v.cpxsh(CPX_IF4B) := '0';
          v.srcinc := "10";
	  if v.ioaddr = '1' then
            v.size := 0;
            v.cpxsh(CPX_IF4B) := '1';
	  else v.size := 7; end if; 
	  v.enable := '1'; v.write := '0'; v.abstate := A_REQ2;
        when STORE_RQ =>
	  v.enable := '1'; v.write := '1'; v.abstate := A_REQ2;
        when INT_RQ =>
	  v.enable := '0'; v.write := '1'; v.abstate := A_REQ2;
        when others =>
	  v.abstate := A_ERR;
-- pragma translate_off
	  assert false report "unhandled PCX packet" &  tost("000"&v.pcxreq) severity failure;
-- pragma translate_on
        end case;
      end if;
    when A_REQ1 =>	-- Set up AHB (integrated in A_IDLE for all except STORE part of CAS/SWAP packets)
      case r.pcxreq is
      when CAS1_RQ | SWAP_RQ =>
	-- This CAS/SWAP code is only for the STORE part as Amba goes straight from IDLE to REQ2 for LOAD part
        -- Reset defaults to what they were before LOAD part of operation executed
	v.srcaddr := v.pcxaddr(31 downto 0);
        if v.pcxsize(1 downto 0) = "11" then
	  v.srcinc := "10";
          if (v.pcxsize(2) = '1') then v.size := 3;
	  else v.size := 1; end if;
        else 
	  v.srcinc := v.pcxsize(1 downto 0); v.size := 0;
        end if;
	v.cnt := 0;
	if r.atomstore = '1' then
	  v.enable := '1'; v.write := '1';
	end if;
	v.abstate := A_REQ2;
      when others =>
	null;
-- pragma translate_off
	assert false report "state is A_REQ1 but pcx request is neither CAS or SWAP, should not happen" &  tost("000"&v.pcxreq) severity failure;
-- pragma translate_on
      end case;
    when A_REQ2 =>	-- Wait for AHB ready
      if v.atom2 = '1' then -- STORE part of CAS or SWAP
	if ( (dmao.ready  = '1') and (r.size = r.cnt) ) or r.atomstore = '0' then
	  -- All info needed for the two atomic packets has been generated, send packets
	  v.abstate := A_REQ3;
	  v.cpxrdy := '1'; -- To send atomic packets
	end if;
      elsif (v.pcxreq = LOAD_RQ) or (v.pcxreq = CAS1_RQ) or (v.pcxreq = SWAP_RQ) then
	-- LOAD or first part of CAS or SWAP
	v.cpxtype := LOAD_RET;	-- LOAD
	if (dmao.ready  = '1') and (r.size = r.cnt) then
	  v.abstate := A_REQ3;
	  -- Normal operations sends a normal cpx packet
	  -- Atomic operations will send invalidate D$ packet
	  v.cpxrdy := '1';
	end if;
      elsif v.pcxreq = IMISS_RQ then
	v.cpxtype := IFILL_RET;	-- IFILL, IO
	if (dmao.ready  = '1') and (r.size = r.cnt) then
	  v.abstate := A_REQ3; v.cpxrdy := '1';
	end if;
      elsif v.pcxreq = STORE_RQ then
	v.cpxtype := ST_ACK;	-- STORE
	-- Cpxdata contains invalidate information for L1$
	-- All zeroes -> no invalidation of either I$ or D$
	v.cpxdata := (others => '0');
	if (dmao.ready  = '1') and (r.size = r.cnt) then
	  v.abstate := A_REQ3; v.cpxrdy := '1';
	end if;
      elsif v.pcxreq = INT_RQ then
	v.cpxtype := INT_RET;	-- Return acknowledge on INT
	v.abstate := A_REQ3; v.cpxrdy := '1';
      else
	v.abstate := A_IDLE;
      end if;
    when A_REQ3 =>		-- Send cpx packet(s) to processor (if first of atomic packets, do not send, just generate packet)
      v.abstate := A_IDLE;
      v.packettypecpx := C_NORMAL;
      if r.atom2 = '1' then -- Store part of atomic packet
	v.atom2 := '0';
      elsif r.cpxtype = ST_ACK then	-- STORE or STORE-part of CAS/SWAP packets
	-- Cpxdata for ST_ACK contains invalidate information for L1$
	-- All zeroes => no invalidation
	v.cpxdata := (others => '0');
      elsif r.cpxtype = INT_RET then	-- INT
        v.cpxnc := r.pcxnc; -- Flush bit
        -- Return incomming VINT
        -- VINT is duplicated
        v.cpxdata(127 downto 82) := (others => '0');
	v.cpxdata(81 downto 64)  := v.pcxdata(17 downto 0); -- VINT
        v.cpxdata(63 downto 18)  := (others => '0');
	v.cpxdata(17 downto 0)   := v.pcxdata(17 downto 0); -- VINT
      else -- LOAD_RET or IFILL_RET
        case r.size is
        when 0 =>
          case r.srcinc is
          when "00" => -- 1 Byte
            case r.srcaddr(1 downto 0) is
	    when "00" => rdata(7 downto 0) :=  r.data(0)(31 downto 24);
	    when "01" => rdata(7 downto 0) :=  r.data(0)(23 downto 16);
	    when "10" => rdata(7 downto 0) :=  r.data(0)(15 downto 8);
	    when others => rdata(7 downto 0) :=  r.data(0)(7 downto 0);
            end case;
	    for i in 0 to 15 loop 
	      v.cpxdata(i*8+7 downto i*8) := rdata(7 downto 0);
	    end loop;
          when "01" => -- 2 Byte
            if r.srcaddr(1) = '1' then rdata := r.data(0)(15 downto 0); end if;
	    for i in 0 to 7 loop 
	      v.cpxdata(i*16+15 downto i*16) := rdata(15 downto 0);
	    end loop;
          when others => -- 4 Byte = 1 Word
            v.cpxdata := r.data(0) & r.data(0) & r.data(0) & r.data(0);
	  end case;
        when 1 => -- 8 Byte = 1 DoubleWord
          v.cpxdata := r.data(0) & r.data(1) & r.data(0) & r.data(1);
        when others => -- 16 Byte = 1 QuadWord
          v.cpxdata := r.data(0) & r.data(1) & r.data(2) & r.data(3);
        end case;
	-- If CAS operation is in progress and word or double word (depending on pcxsize) mismatches, do not perform store
	if (v.pcxreq = CAS1_RQ) and
	 ( (v.pcxsize = "010" and v.pcxdata(63 downto 32) /= v.cpxdata(127 downto 96)) or
	 (v.pcxsize = "011" and (v.pcxdata /= v.cpxdata(127 downto 64) ) ) ) then
	  -- Do not perform store as a mismatching cas operation is in progress
	  v.atomstore := '0';
	else
	  -- Perform store for matching CAS and all SWAP operations
	  v.atomstore := '1';
	end if;
	if (r.pcxreq = SWAP_RQ) or (r.pcxreq = CAS1_RQ) then
	  -- Store should be issued to AMBA before sending packets
	  v.abstate := A_REQ1;
	  v.packettypecpx := C_INVDCACHE; -- Invalidate D$ lines
	  v.atom2 := '1';
	end if;

      end if;
      -- 32 bytes IFILL, send two cpx packets. These packets are NOT atomic!
      if (r.pcxreq = IMISS_RQ) and (r.size = 7) then
	if r.packettypecpx /= C_NORMAL then
	  -- First packet
	  v.abstate := A_REQ3; v.cpxrdy := '1'; -- Another cpx-packet needs to be sent
	else
	  -- Second packet
          v.cpxdata := r.data(4) & r.data(5) & r.data(6) & r.data(7);
          v.cpxsh(CPX_IFILL2) := '1';
	end if;
      end if;
    when others =>
        null;
    end case;

    -- AHB DMA engine

    if (r.cnt < r.size) then burst := '1'; else burst := '0'; end if;
    start := r.enable;
    if dmao.active = '1' then
      if r.write = '0' then
	if dmao.ready = '1' then
	  v.data(r.cnt) := ahbreadword(dmao.rdata);
	  if r.cnt = r.size then 
	    v.cnt := 0; v.inhibit := '1'; v.enable := '0';
	  else v.cnt := r.cnt + 1; end if;
	end if;
      else
	if dmao.ready = '1' then
	  if r.cnt = r.size then 
	    v.cnt := 0; v.inhibit := '1'; v.enable := '0';
	  else v.cnt := r.cnt + 1; end if;
	end if;
      end if;
    end if;

    if fread = '1' then v.raddr := r.raddr + 1; end if;
    oldaddr := r.srcaddr(9 downto 0); oldsize := r.srcinc;
    ainc := decode(oldsize);
    newaddr := oldaddr + ainc(3 downto 0);

    if (dmao.active and dmao.ready and burst) = '1' then
      v.srcaddr(9 downto 0) := newaddr;
    end if;

-- Read APB registers

    case apbi.paddr(5 downto 2) is
    when "0000" => regd(4 downto 0) := t1out.rst_ready & '0' & r.abstate;
    when "1000" => regd(24 downto 0) := 
        v.pcxatom & v.pcxreq & v.pcxnc & v.pcxcpu & v.pcxthre &
        v.pcxinv & v.pcxbis & r.pcxsize & r.pcxaddr(39 downto 32);
    when "1001" => regd := r.pcxaddr(31 downto 0);
    when "1010" => regd := r.pcxdata(63 downto 32);
    when "1011" => regd := r.pcxdata(31 downto 0);
    when others => null;
    end case;

-- Write APB registers

    reset_l := '1';
    if (apbi.psel(pindex) and apbi.penable and apbi.pwrite) = '1' then
      case apbi.paddr(5 downto 2) is
      when "0000" => 
        v.abstate := apbi.pwdata(2 downto 0);
      when "0001" => 
	v.doint := apbi.pwdata(31);
	v.intcmd := apbi.pwdata(17 downto 0);
      when "0010" => 
        v.reset_l := apbi.pwdata(0);
        reset_l := apbi.pwdata(0);
      when others => null;
      end case;
    end if;

-- Reset, initialize variables

    if (rst = '0') or (reset_l = '0') then
      v.atom2 := '0';
      v.packettypecpx := C_NONE;
      v.cpxtype := "0000";
      v.cpxsh := (others => '0');
      v.cpxnc := '0';
      v.cpxpfl := '0';
      v.cpxdata := (others => '0');
      v.enable := '0'; v.write := '0';
      v.cnt  := 0; v.abstate := A_INIT; v.pbstate := P_INIT;
      v.data := (others => (others => '0'));
      v.ld2op := '0'; v.ld2val := '0'; v.pcxnc := '0';
      v.reset_l := '0'; v.doint := '0'; 
    end if;

-- Support for atomic cpx packets

    -- Check if this is the first of two atomic CPX-packets
    -- All atomic packets consists of 1 atomic LOAD_RET followed by 1 ST_ACK
    if (r.packettypecpx = C_NORMAL) and (r.cpxtype = LOAD_RET) and (r.cpxsh(CPX_ATOMIC) = '1') then
      -- Yes, next packet to be sent is the second packet of the atomic series
      v.packettypecpx := C_ATOMIC2;
      -- Save thread info for next packet
      v.oldpcxthre := r.pcxthre;
    end if;

    dmai.address <= r.srcaddr; -- Address;
    if (r.cnt mod 2) = 0 then dmai.wdata <= ahbdrivedata(r.storedata(63 downto 32));
    else dmai.wdata <= ahbdrivedata(r.storedata(31 downto 0)); end if;
--    dmai.wdata   <= r.storedata((31 + 32*(1-(r.cnt mod 2))) downto 32*(1-(r.cnt mod 2)));
    dmai.start   <= start and not v.inhibit;
    dmai.burst   <= burst;
    dmai.busy    <= '0';
    dmai.irq     <= '0';
    dmai.write   <= v.write;
    dmai.size    <= '0' & r.srcinc;

    apbo.prdata  <= regd;
    apbo.pirq    <= (others =>'0');
    apbo.pindex  <= pindex;
    apbo.pconfig <= pconfig;

    -- Send CPX packets
    t1in.cpx_spc_data_rdy_cx2 <= r.cpxrdy;
    case r.packettypecpx is
    when C_ATOMIC2 =>
      t1in.cpx_spc_data_cx2(144 downto 128) <=
	"1" &		-- 144     : Valid
	ST_ACK &	-- 143-140 : type
	"0" &		-- 139     : L2miss
	"00" &		-- 138-137 : ERR
	"1" &		-- 136     : Non-cachable
	r.oldpcxthre &	-- 135-134 : shared: Thread Id in most cases, core has one thread => report the same back
	"00001" &	-- 133-129 : shared, LSB is CPX_ATOMIC
	"0";		-- 128     : reserved, used for "Prefetch Load"
      t1in.cpx_spc_data_cx2(127 downto 0) <=
	(others => '0');-- 127-0   : invalidation vector, all zeroes => no invalidation
    when C_INVDCACHE =>
      t1in.cpx_spc_data_cx2(144 downto 128) <=
	"1" &           -- 144     : Valid
	ST_ACK &        -- 143-140 : type
	"0" &		-- 139     : L2miss
	"00" &		-- 138-137 : ERR
	"1" &           -- 136     : Non-cachable
	"00" &	        -- 135-134 : shared: Thread Id, does not matter on invalidate packet
	"00000" &       -- 133-129 : shared
	"0";	        -- 128     : reserved, used for "Prefetch Load"
      t1in.cpx_spc_data_cx2(127 downto 0) <= (others => '0');
      t1in.cpx_spc_data_cx2(123) <= '1'; -- invalidate all ways of D$ matching index below
      t1in.cpx_spc_data_cx2(122 downto 121) <= v.pcxaddr(5 downto 4);
      t1in.cpx_spc_data_cx2(117 downto 112) <= v.pcxaddr(11 downto 6);
    when C_NORMAL => -- normal packet or atomic packet 1/2
      t1in.cpx_spc_data_cx2 <=
	"1" &		-- 144     : Valid
	r.cpxtype &     -- 143-140 : type
	"0" &		-- 139     : L2miss
	"00" &		-- 138-137 : ERR
	r.cpxnc &       -- 136     : Non-cachable
	r.pcxthre &	-- 135-134 : shared: Thread Id in most cases, core has one thread => report the same back
	r.cpxsh &       -- 133-129 : shared
	r.cpxpfl &	-- 128     : reserved, used for "Prefetch Load"
	r.cpxdata;      -- 127-0   : Data
    when others =>
      t1in.cpx_spc_data_cx2 <= (others => '0');
    end case;

    t1in.reset_l <= r.reset_l;
    t1in.pcx_spc_grant_px <= r.pcxgnt;

    rin <= v;

  end process;


  ahbif : ahbmst generic map (hindex => hindex, venid => VENDOR_SUN,
	devid => SUN_T1, incaddr => 1) 
	port map (rst, clk, dmai, dmao, ahbi, ahbo);


  regs : process(clk)
  begin if rising_edge(clk) then r <= rin; end if; end process;

-- pragma translate_off
    bootmsg1 : report_version 
    generic map ("t1core" & tost(hindex) & 
	": Niagara T1 SPARCV9 core, rev " & tost(0));
    bootmsg2 : report_version 
    generic map ("pcxahb" & tost(pindex) & 
	": PCX/AHB bridge rev " & tost(0) & ", irq " & tost(pirq));
-- pragma translate_on

end;
