-- GAISLER_LICENSE
-----------------------------------------------------------------------------   
-- Entity:      pcxahb
-- File:        pcxahb.vhd
-- Author:      Jiri Gaisler - Gaisler Research
-- Description: PCX/AHB bridge for T1 processor
------------------------------------------------------------------------------  

library ieee;
use ieee.std_logic_1164.all;
library grlib;
use grlib.amba.all;
use grlib.stdlib.all;
use grlib.devices.all;
library gaisler;
use gaisler.misc.all;
use work.t1.all;
library techmap;
use techmap.gencomp.all;

entity pcxahb is
   generic (
     hindex : integer := 0;
     pindex : integer := 0;
     paddr  : integer := 0;
     pmask  : integer := 16#fff#;
     pirq   : integer := 0;
     pow    : integer := 1	-- start on power-on
	);
   port (
      rst   : in  std_logic;
      clk   : in  std_ulogic;
      t1in  : out t1_in_type;
      t1out : in  t1_out_type;
      apbi  : in  apb_slv_in_type;
      apbo  : out apb_slv_out_type;
      ahbi  : in  ahb_mst_in_type;
      ahbo  : out ahb_mst_out_type 
      );
end;      

architecture struct of pcxahb is

constant pconfig : apb_config_type := (
  0 => ahb_device_reg ( VENDOR_GAISLER, GAISLER_T1AHB, 0, 0, pirq),
  1 => apb_iobar(paddr, pmask));

constant dbuf   : integer := 8;

constant PCX_LOAD  : std_logic_vector(4 downto 0) := "00000";
constant PCX_STORE : std_logic_vector(4 downto 0) := "00001";
constant PCX_IFILL : std_logic_vector(4 downto 0) := "10000";
constant PCX_INT   : std_logic_vector(4 downto 0) := "01001";

constant CPX_STORE : std_logic_vector(3 downto 0) := "0100";
constant CPX_INT   : std_logic_vector(3 downto 0) := "0111";

type dma_state_type is (readc, writec);
subtype word32 is std_logic_vector(31 downto 0);
type datavec is array (0 to dbuf-1) of word32;
subtype pcxvec is std_logic_vector(124 downto 0);
constant fdepth : integer := 4;  -- depth of pcx fifo, must be 2**n aligned
type pcxarr is array (0 to fdepth-1) of pcxvec;
constant fbits : integer := log2x(fdepth);

--type bus_state_type is (init, int, idle, req1, req2, req3, err);
constant init  : std_logic_vector(2 downto 0) := "000";
constant int   : std_logic_vector(2 downto 0) := "001";
constant idle  : std_logic_vector(2 downto 0) := "010";
constant req1  : std_logic_vector(2 downto 0) := "011";
constant req2  : std_logic_vector(2 downto 0) := "100";
constant req3  : std_logic_vector(2 downto 0) := "101";
constant err   : std_logic_vector(2 downto 0) := "110";


type reg_type is record
  bstate  : std_logic_vector(2 downto 0); --bus_state_type;
  cpxval  : std_logic;
  cpxtype : std_logic_vector(3 downto 0);
  cpxdata : std_logic_vector(127 downto 0);
  cpxrdy  : std_logic;
  cpxfill2: std_logic;
  cpxnc   : std_logic;  -- not cacheable

  pcxrq   : std_logic_vector(4 downto 0);
  pcxreq  : std_logic_vector(4 downto 0);
  pcxgnt  : std_logic_vector(4 downto 0);
  pcxdata : std_logic_vector(63 downto 0);
  pcxaddr : std_logic_vector(39 downto 0);
  pcxsize : std_logic_vector(2 downto 0);
  pcxcpu  : std_logic_vector(2 downto 0);
  pcxthre : std_logic_vector(1 downto 0);
  pcx4b   : std_logic;
  pcxbis  : std_logic;
  pcxinv  : std_logic;
  pcxatom : std_logic;
  pcxnc   : std_logic;  -- not cacheable
  pcxnew  : std_logic;
  pcxat   : std_logic;

  srcaddr : std_logic_vector(31 downto 0);
  srcinc  : std_logic_vector(1 downto 0);
  len     : std_logic_vector(15 downto 0);
  enable  : std_logic;
  write   : std_logic;
  inhibit : std_logic;
  ld2op   : std_logic;
  ld2val  : std_logic;
  op2     : std_logic_vector(123 downto 0);
  req2    : std_logic_vector(4 downto 0);
  status  : std_logic_vector(1 downto 0);
  data    : datavec;
  cnt, size : integer range 0 to dbuf-1;
  waddr, raddr  : std_logic_vector(1 downto 0);
  pcxbuf : pcxarr;
  reset_l : std_logic;
  doint   : std_logic;
  intcmd  : std_logic_vector(17 downto 0);
end record;

signal r, rin : reg_type;
signal dmai : ahb_dma_in_type;
signal dmao : ahb_dma_out_type;

constant CPX_DA_HI : integer := 127;	-- high data index
constant CPX_VLD   : integer := 144;	-- packet valid

begin

  comb : process(apbi, dmao, rst, r, t1out)
  variable v       : reg_type;
  variable regd    : std_logic_vector(31 downto 0);   -- data from registers
  variable start   : std_logic;
  variable burst   : std_logic;
  variable write   : std_logic;
  variable ready   : std_logic;
  variable retry   : std_logic;
  variable mexc    : std_logic;
  variable irq     : std_logic;
  variable size    : std_logic_vector( 1 downto 0);   -- DMA transfer size
  variable newlen  : std_logic_vector(15 downto 0);
  variable oldaddr : std_logic_vector(9 downto 0);
  variable newaddr : std_logic_vector(9 downto 0);
  variable oldsize : std_logic_vector( 1 downto 0);
  variable ainc    : std_logic_vector( 3 downto 0);
  variable empty   : std_logic;
  variable fspc_pcx_data_pa  : std_logic_vector(124 downto 0);
  variable fread   : std_logic;
  variable rdata   : std_logic_vector(15 downto 0);

  begin

    v := r; regd := (others => '0'); burst := '0'; start := '0';
    write := '0'; ready := '0'; mexc := '0';
    irq := '0'; v.inhibit := '0';
    newlen := r.len - 1;
    v.cpxfill2 := '0';
    rdata :=  r.data(0)(31 downto 16);
    
    -- PCX fifo with pass-through
    if (t1out.spc_pcx_req_pq /= "00000") then 
      v.pcxnew := '1'; v.pcxat := t1out.spc_pcx_atom_pq; v.pcxrq := t1out.spc_pcx_req_pq; 
    else v.pcxnew := '0'; v.pcxat := '0'; v.pcxrq := (others => '0'); end if;
    if r.pcxnew = '1' then 
      v.pcxbuf(conv_integer(r.waddr(fbits-1 downto 0))) := r.pcxat & t1out.spc_pcx_data_pa;
      v.waddr := r.waddr + 1;
      v.pcxgnt := r.pcxrq;
    else v.pcxgnt := "00000"; end if;
    if (r.waddr = r.raddr) then empty := '1'; fspc_pcx_data_pa := r.pcxat & t1out.spc_pcx_data_pa;
    else empty := '0'; fspc_pcx_data_pa := r.pcxbuf(conv_integer(r.raddr(fbits-1 downto 0))); end if;
    fread := '0'; v.cpxrdy := '0'; v.cpxval := '0'; 

    case r.bstate is
    when init =>	-- power-on reset
      if pow = 1 then v.reset_l := '1'; end if; -- start on power-on
      v.waddr := (others => '0'); v.raddr := (others => '0');
      if (t1out.rst_ready = '1') then 
	v.bstate := int; v.cpxrdy := '1';
	v.cpxdata(17 downto 0) := "01" & X"0001";
      end if;
    when int  =>	-- send INT packet to processor
      if r.cpxrdy = '0' then v.cpxrdy := '1'; else
	v.bstate := idle; v.cpxval := '1';  v.cpxrdy := '0';
	v.cpxdata(127 downto 18) := (others => '0');
	v.cpxtype := CPX_INT;	-- INT
      end if;
    when idle =>	-- look for new pcx packet
      if r.doint = '1' then
	v.bstate := int; v.cpxrdy := '1';
	v.cpxdata(17 downto 0) := r.intcmd;
      elsif (empty = '0') or (r.pcxnew = '1') then v.bstate := req1; 
        v.pcxatom := fspc_pcx_data_pa(124); --spc_pcx_atom_pq;
        v.pcxreq  := fspc_pcx_data_pa(122 downto 118);
        v.pcxnc   := fspc_pcx_data_pa(117);
        v.pcxcpu  := fspc_pcx_data_pa(116 downto 114);
        v.pcxthre := fspc_pcx_data_pa(113 downto 112);
        v.pcxinv  := fspc_pcx_data_pa(111);
        v.pcxbis  := fspc_pcx_data_pa(109);
        v.pcxsize := fspc_pcx_data_pa(106 downto 104);
        v.pcxaddr := fspc_pcx_data_pa(103 downto 64);
        v.srcaddr := fspc_pcx_data_pa(95 downto 64);
        v.pcxdata := fspc_pcx_data_pa(63 downto 0);
        v.cnt := 0; fread := '1'; v.pcx4b := '0'; 
        if fspc_pcx_data_pa(105 downto 104) = "11" then
	  v.srcinc := "10";
          if (fspc_pcx_data_pa(106) = '1') then v.size := 3;
	  else v.size := 1; end if;
        else 
	  v.srcinc := fspc_pcx_data_pa(105 downto 104); v.size := 0;
        end if;
        case fspc_pcx_data_pa(122 downto 118) is
        when PCX_LOAD =>	
	  if fspc_pcx_data_pa(103) = '0' then 
	    v.size := 3; v.srcinc := "10"; v.srcaddr(3 downto 0) := "0000";
	  end if; 
	  v.enable := '1'; v.write := '0'; v.bstate := req2;
        when PCX_IFILL =>	
          v.pcx4b := '0'; v.srcinc := "10";
	  if fspc_pcx_data_pa(103) = '1' then v.size := 0; v.pcx4b := '1';
	  else v.size := 7; end if; 
	  v.enable := '1'; v.write := '0'; v.bstate := req2;
        when PCX_STORE =>
	  v.enable := '1'; v.write := '1'; v.bstate := req2;
        when PCX_INT =>
	  v.enable := '0'; v.write := '1'; v.bstate := req2;
        when others =>
	  v.bstate := err;
-- pragma translate_off
	  assert false report "unhandled PCX packet" &  tost("000"&fspc_pcx_data_pa(122 downto 118)) severity failure;
-- pragma translate_on
        end case;
      end if;
    when req2 =>	-- wait for AHB ready
      v.cpxnc := r.srcaddr(31);
      case r.pcxreq is
      when PCX_LOAD =>
	v.cpxtype := "0000";	-- LOAD
	if (dmao.ready  = '1') and (r.size = r.cnt) then
	  v.bstate := req3; v.cpxrdy := '1';
	end if;
      when PCX_IFILL =>
	v.cpxtype := "0001";	-- IFILL, IO
	if (dmao.ready  = '1') and (r.size = r.cnt) then
	  v.bstate := req3; v.cpxrdy := '1';
	end if;
      when PCX_STORE =>
	v.cpxtype := CPX_STORE;	-- STORE
	if (dmao.ready  = '1') and (r.size = r.cnt) then
	  v.bstate := req3; v.cpxrdy := '1';
	end if;
      when PCX_INT =>
        v.cpxnc := r.pcxnc;
	v.cpxtype := CPX_INT;	-- Return acknowledge on INT
	v.bstate := req3; v.cpxrdy := '1';
      when others =>
	v.bstate := idle;
      end case;
    when req3 =>		-- send cpx packet(s) to processor
      v.bstate := idle; v.cpxval := '1';
      if r.cpxtype = CPX_STORE then	-- STORE
	v.cpxdata(127 downto 64) := (others => '0');
	v.cpxdata(63 downto 0) := r.pcxdata;
      elsif r.cpxtype = CPX_INT then	-- INT
	v.cpxdata(127 downto 126) := (others => '0');
	v.cpxdata(125) := r.pcxbis;
	v.cpxdata(124 downto 123) := "00";
	v.cpxdata(122 downto 121) := r.pcxaddr(5 downto 4);
	v.cpxdata(120 downto 118) := r.pcxcpu;
	v.cpxdata(117 downto 112) := r.pcxaddr(11 downto 6);
        v.cpxdata(111 downto 0)   := (others => '0');
      else
        case r.size is
        when 0 =>
          case r.srcinc is
          when "00" => 
            case r.srcaddr(1 downto 0) is
	    when "00" => rdata(7 downto 0) :=  r.data(0)(31 downto 24);
	    when "01" => rdata(7 downto 0) :=  r.data(0)(23 downto 16);
	    when "10" => rdata(7 downto 0) :=  r.data(0)(15 downto 8);
	    when others => rdata(7 downto 0) :=  r.data(0)(7 downto 0);
            end case;
	    for i in 0 to 15 loop 
	      v.cpxdata(i*8+7 downto i*8) := rdata(7 downto 0);
	    end loop;
          when "01" => 
            if r.srcaddr(1) = '1' then rdata := r.data(0)(15 downto 0); end if;
	    for i in 0 to 7 loop 
	      v.cpxdata(i*16+15 downto i*16) := rdata(15 downto 0);
	    end loop;
          when others => v.cpxdata := r.data(0) & r.data(0) & r.data(0) & r.data(0);
	  end case;
        when 1 => v.cpxdata := r.data(0) & r.data(1) & r.data(0) & r.data(1);
        when others => v.cpxdata := r.data(0) & r.data(1) & r.data(2) & r.data(3);
        end case;
      end if;
      if (r.pcxreq = PCX_IFILL) and (r.size = 7) then	-- IFILL2
	if r.cpxval = '0' then
	  v.bstate := req3; v.cpxrdy := '1';
	else
          v.cpxdata := r.data(4) & r.data(5) & r.data(6) & r.data(7);
	  v.cpxfill2 := '1';
	end if;
      end if;
    when others =>
        v.cpxval := '0'; --v.cpxtype := "0000"; v.cpxdata := (others => '0');
    end case;

    -- AHB DMA engine

    if (r.cnt < r.size) then burst := '1'; else burst := '0'; end if;
    start := r.enable;
    if dmao.active = '1' then
      if r.write = '0' then
	if dmao.ready = '1' then
	  v.data(r.cnt) := ahbreadword(dmao.rdata);
	  if r.cnt = r.size then 
	    v.cnt := 0; v.inhibit := '1'; v.enable := '0';
	  else v.cnt := r.cnt + 1; end if;
	end if;
      else
	if dmao.ready = '1' then
	  if r.cnt = r.size then 
	    v.cnt := 0; v.inhibit := '1'; v.enable := '0';
	  else v.cnt := r.cnt + 1; end if;
	end if;
      end if;
    end if;

    if fread = '1' then v.raddr := r.raddr + 1; end if;
    oldaddr := r.srcaddr(9 downto 0); oldsize := r.srcinc;
    ainc := decode(oldsize);
    newaddr := oldaddr + ainc(3 downto 0);

    if (dmao.active and dmao.ready and burst) = '1' then
      v.srcaddr(9 downto 0) := newaddr;
    end if;

-- read APB registers

    case apbi.paddr(5 downto 2) is
    when "0000" => regd(4 downto 0) := t1out.rst_ready & '0' & r.bstate;
    when "1000" => regd(24 downto 0) := 
        v.pcxatom & v.pcxreq & v.pcxnc & v.pcxcpu & v.pcxthre &
        v.pcxinv & v.pcxbis & r.pcxsize & r.pcxaddr(39 downto 32);
    when "1001" => regd := r.pcxaddr(31 downto 0);
    when "1010" => regd := r.pcxdata(63 downto 32);
    when "1011" => regd := r.pcxdata(31 downto 0);
    when others => null;
    end case;

-- write APB registers

    if (apbi.psel(pindex) and apbi.penable and apbi.pwrite) = '1' then
      case apbi.paddr(5 downto 2) is
      when "0000" => 
        v.bstate := apbi.pwdata(2 downto 0);
      when "0001" => 
	v.doint := apbi.pwdata(31);
	v.intcmd := apbi.pwdata(17 downto 0);
      when "0010" => 
        v.reset_l := apbi.pwdata(0);
      when others => null;
      end case;
    end if;

    if rst = '0' then
      v.enable := '0'; v.write := '0';
      v.cnt  := 0; v.bstate := init;
      v.cpxval := '0'; v.cpxtype := "0000"; v.cpxdata := (others => '0');
      v.data := (others => (others => '0'));
      v.ld2op := '0'; v.ld2val := '0'; v.cpxnc := '0'; v.pcxnc := '0';
      v.pcx4b := '0'; v.reset_l := '0'; v.doint := '0'; 
    end if;

    rin <= v;

    dmai.address <= r.srcaddr; -- address;
    dmai.wdata   <= ahbdrivedata(r.pcxdata((31 + 32*(1-(r.cnt mod 2))) downto 32*(1-(r.cnt mod 2))));
    dmai.start   <= start and not v.inhibit;
    dmai.burst   <= burst;
    dmai.busy    <= '0';
    dmai.irq     <= '0';
    dmai.write   <= v.write;
    dmai.size    <= '0' & r.srcinc;

    apbo.prdata  <= regd;
    apbo.pirq    <= (others =>'0');
    apbo.pindex  <= pindex;
    apbo.pconfig <= pconfig;

    t1in.reset_l <= r.reset_l;
    t1in.cpx_spc_data_rdy_cx2 <= r.cpxrdy;
    t1in.cpx_spc_data_cx2 <= r.cpxval & r.cpxtype & "000" & r.cpxnc & r.pcxthre &
	"000" & r.pcx4b  & r.cpxfill2 & "0" & r.cpxdata;
    t1in.pcx_spc_grant_px <= r.pcxgnt;

  end process;


  ahbif : ahbmst generic map (hindex => hindex, venid => VENDOR_SUN,
	devid => SUN_T1, incaddr => 1) 
	port map (rst, clk, dmai, dmao, ahbi, ahbo);


  regs : process(clk)
  begin if rising_edge(clk) then r <= rin; end if; end process;

-- pragma translate_off
    bootmsg1 : report_version 
    generic map ("t1core" & tost(hindex) & 
	": Niagara T1 SPARCV9 core, rev " & tost(0));
    bootmsg2 : report_version 
    generic map ("pcxahb" & tost(pindex) & 
	": PCX/AHB bridge rev " & tost(0) & ", irq " & tost(pirq));
-- pragma translate_on

end;
