/* dumpverilog.c -- Dumps memory region as Verilog representation
   or as hex code
   Copyright (C) 2000 Damjan Lampret, lampret@opencores.org

This file is part of OpenRISC 1000 Architectural Simulator.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */
	
/* Verilog dump can be used for stimulating OpenRISC Verilog RTL models. */

#include <stdio.h>
#include <ctype.h>
#include <string.h>

#include "config.h"

#ifdef HAVE_INTTYPES_H
#include <inttypes.h>
#endif

#include "port.h"
#include "arch.h"
#include "sim-config.h"
#include "parse.h"
#include "abstract.h"
#include "opcode/or32.h"
#include "spr_defs.h"
#include "labels.h"
#include "execute.h"
#include "sprs.h"
#include "stats.h"
#include "except.h"
#include "dumpverilog.h"

extern char *or1ksim_ver;
extern char *disassembled;

void dumpverilog(char *verilog_modname, oraddr_t from, oraddr_t to)
{
  unsigned int i, done = 0;
  struct label_entry *tmp;
  char dis[DISWIDTH + 100];  
  uint32_t insn;
  int index;
  PRINTF("// This file was generated by or1ksim version %s\n", or1ksim_ver);
  PRINTF(OR1K_MEM_VERILOG_HEADER(verilog_modname, from/DWQ, to/DWQ, (DISWIDTH*8)));
  
  for(i = from; i < to; i++) {      
    if(!(i & 3)) {
      insn = eval_direct32(i, 0, 0);
      index = insn_decode(insn);
      if (index >= 0) {
          if (verify_memoryarea(i) && (tmp = get_label(i)))
          if (tmp)
            PRINTF("\n//\t%s%s", tmp->name, LABELEND_CHAR);
	  
          PRINTF("\n\tmem['h%x] = %d'h%.8"PRIx32";", i/DWQ, DW,
                 eval_direct32(i, 0, 0));
	  
          disassemble_insn (insn);
          strcpy (dis, disassembled);
	  
          if (strlen(dis) < DISWIDTH)
            memset(dis + strlen(dis), ' ', DISWIDTH);
          dis[DISWIDTH] = '\0';
          PRINTF("\n\tdis['h%x] = {\"%s\"};", i/DWQ, dis);
          dis[0] = '\0';
          i += insn_len(index) - 1;
          done = 1;
          continue;
      }
    }

    if (i % 64 == 0)
      PRINTF("\n");
  
    PRINTF("\n\tmem['h%x] = 'h%.2x;", i/DWQ, eval_direct8(i, 0, 0));
    done = 1;
  }
  
  if (done)
    {
      PRINTF(OR1K_MEM_VERILOG_FOOTER);
      return;
    }
  
  /* this needs to be fixed */
  
  for(i = from; i < to; i++)
    {
      if (i % 8 == 0)
        PRINTF("\n%.8x:  ", i);
      
      /* don't print ascii chars below 0x20. */
      if (eval_direct32(i, 0, 0) < 0x20)
        PRINTF("0x%.2x     ", (uint8_t)eval_direct32(i, 0, 0));
      else
        PRINTF("0x%.2x'%c'  ", (uint8_t)eval_direct32(i, 0, 0),
               (char)eval_direct32(i, 0, 0)); 
    }
  PRINTF(OR1K_MEM_VERILOG_FOOTER);
}

void dumphex(oraddr_t from, oraddr_t to)
{
  oraddr_t i;
  uint32_t insn;
  int index;
  
  for(i = from; i < to; i++) {
    if(!(i & 3)) {
      insn = eval_direct32(i, 0, 0);
      index = insn_decode(insn);
      if(index >= 0) {	
        PRINTF("%.8"PRIx32"\n", eval_direct32(i, 0, 0));
        i += insn_len(index) - 1;
        continue;
      }
    }
    PRINTF("%.2x\n", eval_direct8(i, 0, 0));
  }
}
