/* generate.c -- generates file execgen.c from instruction set
   Copyright (C) 1999 Damjan Lampret, lampret@opencores.org
   Copyright (C) 2005 Gyrgy `nog' Jeney, nog@sdf.lonestar.org

This file is part of OpenRISC 1000 Architectural Simulator.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>

#include "config.h"
#include "opcode/or32.h"

static char *in_file;
static char *out_file;

/* Whether this instruction stores something in register */
static int write_to_reg;

static int out_lines = 0;

void debug(int level, const char *format, ...)
{
#if DEBUG
  char *p;
  va_list ap;

  if ((p = malloc(1000)) == NULL)
    return;
  va_start(ap, format);
  (void) vsnprintf(p, 1000, format, ap);
  va_end(ap);
  printf("%s\n", p);
  fflush(stdout);
  free(p);
#endif
}

static int shift_fprintf(int level, FILE *f, const char *fmt, ...)
{
  va_list ap;
  int i;

  va_start(ap, fmt);
  for(i = 0; i < level; i++)
    fprintf(f, "  ");

  i = vfprintf(f, fmt, ap);
  va_end(ap);

  out_lines++;
  return i + (level * 2);
}

/* Generates a execute sequence for one instruction */
int output_function (FILE *fo, const char *func_name, int level)
{
  FILE *fi;
  int olevel;
  int line_num = 0;

  if ((fi = fopen (in_file, "rt")) == NULL) {
    printf("could not open file\n");
    return 1;
  }

  while (!feof (fi)) {
    char line[10000], *str = line;
    fgets (str, sizeof (line), fi);
    line[sizeof (line) - 1] = 0;
    line_num++;
    if (strncmp (str, "INSTRUCTION (", 13) == 0) {
      char *s;
      str += 13;
      while (isspace (*str)) str++;
      s = str;
      while (*s && *s != ')') s++;
      *s = 0;
      while (isspace(*(s - 1))) s--;
      *s = 0;
      if (strcmp (str, func_name) == 0) {
        olevel = 1;
        str += strlen (str) + 1;
        while (isspace (*str)) str++;
        s = str;
        while (*s && *s != '\n' && *s != '\r') s++;
        *s = 0;
        while (isspace(*(s - 1))) s--;
        *s = 0;
        /*shift_fprintf (level, fo, "#line %i \"%s\"\n", line_num, in_file);*/
        shift_fprintf (level, fo, "%s", str);
        shift_fprintf (level, fo, "   /* \"%s\" */\n", func_name);
        do {
          fgets (line, sizeof (line), fi);
          line[sizeof(line) - 1] = 0;
          for (str = line; *str; str++) {
            if (*str == '{') olevel++;
            else if (*str == '}') olevel--;
          }
          shift_fprintf (level, fo, "%s", line);
        } while (olevel);
	fclose(fi);
        /*shift_fprintf (level, fo, "#line %i \"%s\"\n", out_lines, out_file);*/
        return 0;
      }
    }
  }
  shift_fprintf (level, fo, "%s ();\n", func_name);

  fclose(fi);
  return 0;
}

/* Parses operands. */

static int
gen_eval_operands (FILE *fo, int insn_index, int level)
{
  struct insn_op_struct *opd = op_start[insn_index];
  int i;
  int num_ops;
  int nbits = 0;
  int set_param = 0;
  int dis = 0;
  int sbit;
  int dis_op = -1;

  write_to_reg = 0;

  shift_fprintf (level, fo, "uorreg_t ");

  /* Count number of operands */
  for (i = 0, num_ops = 0;; i++) {
    if (!(opd[i].type & OPTYPE_OP))
      continue;
    if (opd[i].type & OPTYPE_DIS)
      continue;
    if (num_ops)
      fprintf(fo, ", ");
    fprintf(fo, "%c", 'a' + num_ops);
    num_ops++;
    if (opd[i].type & OPTYPE_LAST)
      break;
  }

  fprintf (fo, ";\n");

  shift_fprintf (level, fo, "/* Number of operands: %i */\n", num_ops);

  i = 0;
  num_ops = 0;
  do {
/*
    printf("opd[%i].type<last> = %c\n", i, opd->type & OPTYPE_LAST ? '1' : '0');    printf("opd[%i].type<op> = %c\n", i, opd->type & OPTYPE_OP ? '1' : '0');
    printf("opd[%i].type<reg> = %c\n", i, opd->type & OPTYPE_REG ? '1' : '0');
    printf("opd[%i].type<sig> = %c\n", i, opd->type & OPTYPE_SIG ? '1' : '0');
    printf("opd[%i].type<dis> = %c\n", i, opd->type & OPTYPE_DIS ? '1' : '0');
    printf("opd[%i].type<shr> = %i\n", i, opd->type & OPTYPE_SHR);
    printf("opd[%i].type<sbit> = %i\n", i, (opd->type & OPTYPE_SBIT) >> OPTYPE_SBIT_SHR);
    printf("opd[%i].data = %i\n", i, opd->data);
*/

    if (!nbits)
      shift_fprintf (level, fo, "%c = (insn >> %i) & 0x%x;\n", 'a' + num_ops,
                     opd->type & OPTYPE_SHR, (1 << opd->data) - 1);
    else
      shift_fprintf (level, fo, "%c |= ((insn >> %i) & 0x%x) << %i;\n",
                     'a' + num_ops, opd->type & OPTYPE_SHR,
                     (1 << opd->data) - 1, nbits);

    nbits += opd->data;

    if ((opd->type & OPTYPE_DIS) && (opd->type & OPTYPE_OP)) {
      sbit = (opd->type & OPTYPE_SBIT) >> OPTYPE_SBIT_SHR;
      if (opd->type & OPTYPE_SIG)
        shift_fprintf (level, fo, "if(%c & 0x%08x) %c |= 0x%x;\n",
                       'a' + num_ops, 1 << sbit, 'a' + num_ops,
                       0xffffffff << sbit);
      opd++;
      shift_fprintf (level, fo, "*(orreg_t *)&%c += (orreg_t)cpu_state.reg[(insn >> %i) & 0x%x];\n",
                     'a' + num_ops, opd->type & OPTYPE_SHR,
                     (1 << opd->data) - 1);
      dis = 1;
      dis_op = num_ops;
    }

    if (opd->type & OPTYPE_OP) {
      sbit = (opd->type & OPTYPE_SBIT) >> OPTYPE_SBIT_SHR;
      if (opd->type & OPTYPE_SIG)
        shift_fprintf (level, fo, "if(%c & 0x%08x) %c |= 0x%x;\n",
                       'a' + num_ops, 1 << sbit, 'a' + num_ops,
                       0xffffffff << sbit);
      if ((opd->type & OPTYPE_REG) && !dis) {
        if(!i) {
          shift_fprintf (level, fo, "#define SET_PARAM0(val) cpu_state.reg[a] = val\n");
          set_param = 1;
        }
        shift_fprintf (level, fo, "#define PARAM%i cpu_state.reg[%c]\n", num_ops,
                      'a' + num_ops);
        if(opd->type & OPTYPE_DST)
          write_to_reg = 1;
      } else {
        shift_fprintf (level, fo, "#define PARAM%i %c\n", num_ops,
                       'a' + num_ops);
      }
      num_ops++;
      nbits = 0;
      dis = 0;
    }

    if ((opd->type & OPTYPE_LAST))
      break;
    opd++;
    i++;
  } while (1);

  output_function (fo, or32_opcodes[insn_index].function_name, level);

  if (set_param)
    shift_fprintf (level, fo, "#undef SET_PARAM\n");

  for (i = 0; i < num_ops; i++)
    shift_fprintf (level, fo, "#undef PARAM%i\n", i);

  return dis_op;
}

/* Generates decode and execute for one instruction instance */
static int output_call (FILE *fo, int index, int level)
{
  int dis_op = -1;

  /*printf ("%i:%s\n", index, insn_name (index));*/

  shift_fprintf (level++, fo, "{\n");

  if (index >= 0)
    dis_op = gen_eval_operands (fo, index, level);

  if (index < 0) output_function (fo, "l_invalid", level);

  fprintf (fo, "\n");

  shift_fprintf (level++, fo, "if (do_stats) {\n");

  if (dis_op >= 0)
    shift_fprintf (level, fo, "cpu_state.insn_ea = %c;\n", 'a' + dis_op);

  shift_fprintf (level, fo, "current->insn_index = %i;   /* \"%s\" */\n", index,
                 insn_name (index));

  shift_fprintf (level, fo, "analysis(current);\n");
  shift_fprintf (--level, fo, "}\n");

  if (write_to_reg)
    shift_fprintf (level, fo, "cpu_state.reg[0] = 0; /* Repair in case we changed it */\n");
  shift_fprintf (--level, fo, "}\n");
  return 0;
}

/* Generates .c file header */
static int generate_header (FILE *fo)
{
  fprintf (fo, "/* This file was automatically generated by generate (see cpu/or32/generate.c) */\n\n");
  fprintf (fo, "static inline void decode_execute (struct iqueue_entry *current)\n{\n");
  fprintf (fo, "  uint32_t insn = current->insn;\n");
  out_lines = 5;
  return 0;
}

/* Generates .c file footer */
static int generate_footer (FILE *fo)
{
  fprintf (fo, "}\n");
  return 0;
}

/* Decodes all instructions and generates code for that.  This function
   is similar to insn_decode, except it decodes all instructions. */
static int generate_body (FILE *fo, unsigned long *a, unsigned long cur_mask, int level)
{
  unsigned long shift = *a;
  unsigned long mask;
  int i;
  int prev_inv = 0;

  if (!(*a & LEAF_FLAG)) {
    shift = *a++;
    mask = *a++;
    shift_fprintf (level, fo, "switch((insn >> %i) & 0x%x) {\n", shift,
                   mask);
    for (i = 0; i <= mask; i++, a++) {
      if (!*a) {
        shift_fprintf (level, fo, "case 0x%x:\n", i);
        prev_inv = 1;
      } else {
        if(prev_inv) {
          shift_fprintf (++level, fo, "/* Invalid instruction(s) */\n");
          shift_fprintf (level--, fo, "break;\n");
        }
        shift_fprintf (level, fo, "case 0x%x:\n", i);
        generate_body (fo, automata + *a, cur_mask | (mask << shift), ++level);
        shift_fprintf (level--, fo, "break;\n");
        prev_inv = 0;
      }
    }
    if (prev_inv) {
      shift_fprintf (++level, fo, "/* Invalid instruction(s) */\n");
      shift_fprintf (level--, fo, "break;\n");
    }
    shift_fprintf (level, fo, "}\n");
  } else {
    i = *a & ~LEAF_FLAG;

    /* Final check - do we have direct match?
       (based on or32_opcodes this should be the only possibility,
       but in case of invalid/missing instruction we must perform a check)  */

    if (ti[i].insn_mask != cur_mask) {
      shift_fprintf (level, fo, "/* Not unique: real mask %08lx and current mask %08lx differ - do final check */\n", ti[i].insn_mask, cur_mask);
      shift_fprintf (level++, fo, "if((insn & 0x%x) == 0x%x) {\n",
                     ti[i].insn_mask, ti[i].insn);
    }
    shift_fprintf (level, fo, "/* Instruction: %s */\n", or32_opcodes[i].name);

    output_call (fo, i, level);

    if (ti[i].insn_mask != cur_mask) {
      shift_fprintf (--level, fo, "} else {\n");
      shift_fprintf (++level, fo, "/* Invalid insn */\n");
      output_call (fo, -1, level);
      shift_fprintf (--level, fo, "}\n");
    }
  }
  return 0;
}
      
/* Main function; it takes two parameters:
   input_file(possibly insnset.c) output_file(possibly execgen.c)*/
int main (int argc, char *argv[])
{
  FILE *fo;
 
  if (argc != 3) {
    fprintf (stderr, "USAGE: generate input_file(possibly insnset.c) output_file(possibly execgen.c)\n");
    exit (-1);
  }
  
  in_file = argv[1];
  out_file = argv[2];
  if (!(fo = fopen (argv[2], "wt+"))) {
    fprintf (stderr, "Cannot create '%s'.\n", argv[2]);
    exit (1);
  }

  build_automata ();
  if (generate_header (fo)) {
    fprintf (stderr, "generate_header\n");
    return 1;
  }

  if (generate_body (fo, automata, 0, 1)) {
    fprintf (stderr, "generate_body\n");
    return 1;
  }

  if (generate_footer (fo)) {
    fprintf (stderr, "generate_footer\n");
    return 1;
  }

  fclose (fo);
  destruct_automata ();
  return 0;
}

