/* Table of opcodes for the OpenRISC 1000 ISA.
   Copyright 1990, 1991, 1992, 1993 Free Software Foundation, Inc.
   Contributed by Damjan Lampret (lampret@opencores.org).
   
This file is part of or1k_gen_isa, or1ksim, GDB and GAS.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

/* We treat all letters the same in encode/decode routines so
   we need to assign some characteristics to them like signess etc.*/

#ifndef OR32_H_ISA
#define OR32_H_ISA

#define NUM_UNSIGNED (0)
#define NUM_SIGNED (1)

#ifndef PARAMS
#define PARAMS(x) x
#endif

#ifndef CONST
#define CONST const
#endif

#define MAX_GPRS 32
#define PAGE_SIZE 8192
#undef __HALF_WORD_INSN__

#define OPERAND_DELIM (',')

#define OR32_IF_DELAY (1)
#define OR32_W_FLAG   (2)
#define OR32_R_FLAG   (4)

#if defined(HAS_EXECUTION)
# if SIMPLE_EXECUTION
#  include "simpl32_defs.h"
# elif DYNAMIC_EXECUTION
#  include "dyn32_defs.h"
# else
extern void l_none PARAMS((void));
# endif
#else
extern void l_none PARAMS((void));
#endif


struct or32_letter {
  char letter;
  int  sign;
  /* int  reloc; relocation per letter ??*/
};

enum insn_type {
 it_unknown,
 it_exception,
 it_arith,
 it_shift,
 it_compare,
 it_branch,
 it_jump,
 it_load,
 it_store,
 it_movimm,
 it_move,
 it_extend,
 it_nop,
 it_mac,
 it_float };

/* Main instruction specification array.  */
struct or32_opcode {
  /* Name of the instruction.  */
  char *name;

  /* A string of characters which describe the operands.
     Valid characters are:
     ,() Itself.  Characters appears in the assembly code.
     rA	 Register operand.
     rB  Register operand.
     rD  Register operand (destination).
     I	 An immediate operand, range -32768 to 32767.
     J	 An immediate operand, range . (unused)
     K	 An immediate operand, range 0 to 65535.
     L	 An immediate operand, range 0 to 63.
     M	 An immediate operand, range . (unused)
     N	 An immediate operand, range -33554432 to 33554431.
     O	 An immediate operand, range . (unused) */
  char *args;

  /* Opcode and operand encoding. */
  char *encoding;

#ifdef HAS_EXECUTION
# if COMPLEX_EXECUTION
  char *function_name;
# elif SIMPLE_EXECUTION
  void (*exec)(struct iqueue_entry *);
# else /* DYNAMIC_EXECUTION */
  void (*exec)(struct op_queue *opq, int param_t[3], orreg_t param[3], int);
# endif
#else  /* HAS_EXECUTION */
  void (*exec)(void);
#endif

  unsigned int flags;
  enum insn_type func_unit;
};

/* This operand is the last in the list */
#define OPTYPE_LAST (0x80000000)
/* This operand marks the end of the operand sequence (for things like I(rD)) */
#define OPTYPE_OP   (0x40000000)
/* The operand specifies a register index */
#define OPTYPE_REG  (0x20000000)
/* The operand must be sign extended */
#define OPTYPE_SIG  (0x10000000)
/* Operand is a relative address, the `I' in `I(rD)' */
#define OPTYPE_DIS  (0x08000000)
/* The operand is a destination */
#define OPTYPE_DST  (0x04000000)
/* Which bit of the operand is the sign bit */
#define OPTYPE_SBIT (0x00001F00)
/* Amount to shift the instruction word right to get the operand */
#define OPTYPE_SHR  (0x0000001F)
#define OPTYPE_SBIT_SHR (8)

/* MM: Data how to decode operands.  */
extern struct insn_op_struct {
  unsigned long type;
  unsigned long data;
} **op_start;

/* Leaf flag used in automata building */
#define LEAF_FLAG         (0x80000000)

struct temp_insn_struct
{
  unsigned long insn;
  unsigned long insn_mask;
  int in_pass;
};

extern unsigned long *automata;
extern struct temp_insn_struct *ti;

extern CONST struct or32_letter or32_letters[];

extern CONST struct  or32_opcode or32_opcodes[];

extern CONST int num_opcodes;

/* Calculates instruction length in bytes.  Always 4 for OR32. */
extern int insn_len PARAMS((int insn_index));

/* Is individual insn's operand signed or unsigned? */
extern int letter_signed PARAMS((char l));

/* Number of letters in the individual lettered operand. */
extern int letter_range PARAMS((char l));

/* MM: Returns index of given instruction name.  */
extern int insn_index PARAMS((char *insn));

/* MM: Returns instruction name from index.  */
extern CONST char *insn_name PARAMS ((int index));

/* MM: Constructs new FSM, based on or32_opcodes.  */ 
extern void build_automata PARAMS ((void));

/* MM: Destructs FSM.  */ 
extern void destruct_automata PARAMS ((void));

/* MM: Decodes instruction using FSM.  Call build_automata first.  */
extern int insn_decode PARAMS((unsigned int insn));

/* Disassemble one instruction from insn to disassemble.
   Return the size of the instruction.  */
int disassemble_insn (unsigned long insn);

/* Disassemble one instruction from insn index.
   Return the size of the instruction.  */
int disassemble_index (unsigned long insn, int index);

/* FOR INTERNAL USE ONLY */
/* Automatically does zero- or sign- extension and also finds correct
   sign bit position if sign extension is correct extension. Which extension
   is proper is figured out from letter description. */
unsigned long extend_imm(unsigned long imm, char l);

/* Extracts value from opcode */
unsigned long or32_extract(char param_ch, char *enc_initial, unsigned long insn);

#endif

